import * as z from "zod";
import { defineAction } from "./action.js";
import { GenkitError } from "./error.js";
import { toJsonSchema } from "./schema.js";
const OperationSchema = z.object({
  action: z.string().optional(),
  id: z.string(),
  done: z.boolean().optional(),
  output: z.any().optional(),
  error: z.object({ message: z.string() }).passthrough().optional(),
  metadata: z.record(z.string(), z.any()).optional()
});
async function lookupBackgroundAction(registry, key) {
  const root = await registry.lookupAction(key);
  if (!root) return void 0;
  const actionName = key.substring(key.indexOf("/", 1) + 1);
  return new BackgroundActionImpl(
    root,
    await registry.lookupAction(`/check-operation/${actionName}/check`),
    await registry.lookupAction(`/cancel-operation/${actionName}/cancel`)
  );
}
class BackgroundActionImpl {
  __action;
  startAction;
  checkAction;
  cancelAction;
  constructor(startAction, checkAction, cancelAction) {
    this.__action = {
      name: startAction.__action.name,
      description: startAction.__action.description,
      inputSchema: startAction.__action.inputSchema,
      inputJsonSchema: startAction.__action.inputJsonSchema,
      metadata: startAction.__action.metadata,
      actionType: startAction.__action.actionType
    };
    this.startAction = startAction;
    this.checkAction = checkAction;
    this.cancelAction = cancelAction;
  }
  async start(input, options) {
    return await this.startAction(input, options);
  }
  async check(operation) {
    return await this.checkAction(operation);
  }
  get supportsCancel() {
    return !!this.cancelAction;
  }
  async cancel(operation) {
    if (!this.cancelAction) {
      return operation;
    }
    return await this.cancelAction(operation);
  }
}
function defineBackgroundAction(registry, config) {
  const startAction = defineAction(
    registry,
    {
      actionType: config.actionType,
      name: config.name,
      description: config.description,
      inputSchema: config.inputSchema,
      inputJsonSchema: config.inputJsonSchema,
      outputSchema: OperationSchema,
      metadata: {
        ...config.metadata,
        outputSchema: toJsonSchema({
          schema: config.outputSchema,
          jsonSchema: config.outputJsonSchema
        })
      },
      use: config.use
    },
    async (input, options) => {
      const operation = await config.start(input, options);
      operation.action = `/${config.actionType}/${config.name}`;
      return operation;
    }
  );
  const checkAction = defineAction(
    registry,
    {
      actionType: "check-operation",
      name: `${config.name}/check`,
      description: config.description,
      inputSchema: OperationSchema,
      inputJsonSchema: config.inputJsonSchema,
      outputSchema: OperationSchema,
      metadata: {
        ...config.metadata,
        outputSchema: toJsonSchema({
          schema: config.outputSchema,
          jsonSchema: config.outputJsonSchema
        })
      }
    },
    async (input) => {
      const operation = await config.check(input);
      operation.action = `/${config.actionType}/${config.name}`;
      return operation;
    }
  );
  let cancelAction = void 0;
  if (config.cancel) {
    cancelAction = defineAction(
      registry,
      {
        actionType: "cancel-operation",
        name: `${config.name}/cancel`,
        description: config.description,
        inputSchema: OperationSchema,
        inputJsonSchema: config.inputJsonSchema,
        outputSchema: OperationSchema,
        metadata: {
          ...config.metadata,
          outputSchema: toJsonSchema({
            schema: config.outputSchema,
            jsonSchema: config.outputJsonSchema
          })
        }
      },
      async (input) => {
        if (!config.cancel) {
          throw new GenkitError({
            status: "UNAVAILABLE",
            message: `${config.name} does not support cancellation.`
          });
        }
        const operation = await config.cancel(input);
        operation.action = `/${config.actionType}/${config.name}`;
        return operation;
      }
    );
  }
  return new BackgroundActionImpl(startAction, checkAction, cancelAction);
}
export {
  OperationSchema,
  defineBackgroundAction,
  lookupBackgroundAction
};
//# sourceMappingURL=background-action.mjs.map