import { ExecutablePrompt, InterruptConfig, ToolAction, GenerationCommonConfigSchema, GenerateOptions, GenerateResponseData } from '@genkit-ai/ai';
import { ChatOptions, Chat } from '@genkit-ai/ai/chat';
import { SessionOptions, Session } from '@genkit-ai/ai/session';
import '@genkit-ai/ai/tool';
import { z, Operation } from '@genkit-ai/core';
import { Formatter } from '@genkit-ai/ai/formats';
import { G as Genkit, a as GenkitOptions } from './index-D3afp1pc.mjs';
import '@genkit-ai/ai/reranker';
import '@genkit-ai/ai/embedder';
import '@genkit-ai/ai/evaluator';
import '@genkit-ai/ai/model';
import '@genkit-ai/ai/retriever';
import '@genkit-ai/core/registry';

/**
 * Copyright 2025 Google LLC
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/**
 * WARNING: these APIs are considered unstable and subject to frequent breaking changes that may not honor semver.
 *
 * Initializes Genkit BETA APIs with a set of options.
 *
 * This will create a new Genkit registry, register the provided plugins, stores, and other configuration. This
 * should be called before any flows are registered.
 *
 * @beta
 */
declare function genkit(options: GenkitOptions): GenkitBeta;
/**
 * Genkit BETA APIs.
 *
 * @beta
 */
declare class GenkitBeta extends Genkit {
    constructor(options?: GenkitOptions);
    /**
     * Create a chat session with the provided options.
     *
     * ```ts
     * const chat = ai.chat({
     *   system: 'talk like a pirate',
     * })
     * let response = await chat.send('tell me a joke')
     * response = await chat.send('another one')
     * ```
     *
     * @beta
     */
    chat<I>(options?: ChatOptions<I>): Chat;
    /**
     * Create a chat session with the provided preabmle.
     *
     * ```ts
     * const triageAgent = ai.definePrompt({
     *   system: 'help the user triage a problem',
     * })
     * const chat = ai.chat(triageAgent)
     * const { text } = await chat.send('my phone feels hot');
     * ```
     *
     * @beta
     */
    chat<I>(preamble: ExecutablePrompt<I>, options?: ChatOptions<I>): Chat;
    /**
     * Create a session for this environment.
     */
    createSession<S = any>(options?: SessionOptions<S>): Session<S>;
    /**
     * Loads a session from the store.
     *
     * @beta
     */
    loadSession(sessionId: string, options: SessionOptions): Promise<Session>;
    /**
     * Gets the current session from async local storage.
     *
     * @beta
     */
    currentSession<S = any>(): Session<S>;
    /**
     * Defines and registers a custom model output formatter.
     *
     * Here's an example of a custom JSON output formatter:
     *
     * ```ts
     * import { extractJson } from 'genkit/extract';
     *
     * ai.defineFormat(
     *   { name: 'customJson' },
     *   (schema) => {
     *     let instructions: string | undefined;
     *     if (schema) {
     *       instructions = `Output should be in JSON format and conform to the following schema:
     * \`\`\`
     * ${JSON.stringify(schema)}
     * \`\`\`
     * `;
     *     }
     *     return {
     *       parseChunk: (chunk) => extractJson(chunk.accumulatedText),
     *       parseMessage: (message) => extractJson(message.text),
     *       instructions,
     *     };
     *   }
     * );
     *
     * const { output } = await ai.generate({
     *   prompt: 'Invent a menu item for a pirate themed restaurant.',
     *   output: { format: 'customJson', schema: MenuItemSchema },
     * });
     * ```
     *
     * @beta
     */
    defineFormat(options: {
        name: string;
    } & Formatter['config'], handler: Formatter['handler']): {
        config: Formatter['config'];
        handler: Formatter['handler'];
    };
    /**
     * Defines and registers an interrupt.
     *
     * Interrupts are special tools that halt model processing and return control back to the caller. Interrupts make it simpler to implement
     * "human-in-the-loop" and out-of-band processing patterns that require waiting on external actions to complete.
     *
     * @beta
     */
    defineInterrupt<I extends z.ZodTypeAny, O extends z.ZodTypeAny>(config: InterruptConfig<I, O>): ToolAction<I, O>;
    /**
     * Starts a generate operation for long running generation models, typically for
     * video and complex audio generation.
     *
     * See {@link GenerateOptions} for detailed information about available options.
     *
     * ```ts
     * const operation = await ai.generateOperation({
     *   model: googleAI.model('veo-2.0-generate-001'),
     *   prompt: 'A banana riding a bicycle.',
     * });
     * ```
     *
     * The status of the operation and final result can be obtained using {@link Genkit.checkOperation}.
     */
    generateOperation<O extends z.ZodTypeAny = z.ZodTypeAny, CustomOptions extends z.ZodTypeAny = typeof GenerationCommonConfigSchema>(opts: GenerateOptions<O, CustomOptions> | PromiseLike<GenerateOptions<O, CustomOptions>>): Promise<Operation<GenerateResponseData>>;
}

export { GenkitBeta, GenkitOptions as GenkitBetaOptions, genkit };
