'use server';

/**
 * @fileOverview An AI agent to generate agent personas and tone of voice.
 *
 * - generateAgentPersona - A function that handles the agent persona generation process.
 * - AgentPersonaInput - The input type for the generateAgentPersona function.
 * - AgentPersonaOutput - The return type for the generateAgentPersona function.
 */

import {ai} from '@/ai/genkit';
import {z} from 'genkit';

const AgentPersonaInputSchema = z.object({
  agentDescription: z
    .string()
    .describe('The overall purpose and functionality of the agent.'),
  creatorPreferences: z
    .string()
    .optional()
    .describe('Any specific preferences or ideas from the creator.'),
});
export type AgentPersonaInput = z.infer<typeof AgentPersonaInputSchema>;

const AgentPersonaOutputSchema = z.object({
  persona: z.string().describe('A detailed persona for the agent.'),
  toneOfVoice: z
    .string()
    .describe('The suggested tone of voice for the agent.'),
});
export type AgentPersonaOutput = z.infer<typeof AgentPersonaOutputSchema>;

export async function generateAgentPersona(
  input: AgentPersonaInput
): Promise<AgentPersonaOutput> {
  return generateAgentPersonaFlow(input);
}

const prompt = ai.definePrompt({
  name: 'agentPersonaPrompt',
  input: {schema: AgentPersonaInputSchema},
  output: {schema: AgentPersonaOutputSchema},
  prompt: `You are an AI expert at crafting agent personas. Given the following information, create a detailed persona and tone of voice for the agent.

Agent Description: {{{agentDescription}}}
Creator Preferences: {{{creatorPreferences}}}

Persona:
Tone of Voice: `,
});

const generateAgentPersonaFlow = ai.defineFlow(
  {
    name: 'generateAgentPersonaFlow',
    inputSchema: AgentPersonaInputSchema,
    outputSchema: AgentPersonaOutputSchema,
  },
  async input => {
    const {output} = await prompt(input);
    return output!;
  }
);
