
'use server';

/**
 * @fileOverview An AI agent to generate a knowledge base from a description.
 *
 * - generateKnowledgeBase - A function that handles the knowledge base generation process.
 * - KnowledgeBaseGeneratorInput - The input type for the generateKnowledgeBase function.
 * - KnowledgeBaseGeneratorOutput - The return type for the generateKnowledgeBase function.
 */

import {ai} from '@/ai/genkit';
import {z} from 'genkit';

const KnowledgeBaseGeneratorInputSchema = z.object({
  agentDescription: z
    .string()
    .describe('The overall purpose and functionality of the agent.'),
});
export type KnowledgeBaseGeneratorInput = z.infer<typeof KnowledgeBaseGeneratorInputSchema>;

const KnowledgeBaseGeneratorOutputSchema = z.object({
  knowledgeBaseText: z
    .string()
    .describe('A starter knowledge base in plain text format.'),
});
export type KnowledgeBaseGeneratorOutput = z.infer<typeof KnowledgeBaseGeneratorOutputSchema>;

export async function generateKnowledgeBase(
  input: KnowledgeBaseGeneratorInput
): Promise<KnowledgeBaseGeneratorOutput> {
  return generateKnowledgeBaseFlow(input);
}

const prompt = ai.definePrompt({
  name: 'knowledgeBaseGeneratorPrompt',
  input: {schema: KnowledgeBaseGeneratorInputSchema},
  output: {schema: KnowledgeBaseGeneratorOutputSchema},
  prompt: `You are an AI assistant that creates a knowledge base for another AI agent. Based on the following agent description, generate a detailed starter knowledge base in plain text. The knowledge base should be comprehensive, well-structured, and provide the agent with the necessary information to perform its tasks effectively.

Agent Description: {{{agentDescription}}}

Generated Knowledge Base:`,
});

const generateKnowledgeBaseFlow = ai.defineFlow(
  {
    name: 'generateKnowledgeBaseFlow',
    inputSchema: KnowledgeBaseGeneratorInputSchema,
    outputSchema: KnowledgeBaseGeneratorOutputSchema,
  },
  async input => {
    const {output} = await prompt(input);
    return output!;
  }
);
