'use server';
/**
 * @fileOverview Summarizes the content of a knowledge base (PDF or article) for creator verification.
 *
 * - summarizeKnowledgeBase - A function that summarizes the knowledge base content.
 * - KnowledgeBaseSummarizerInput - The input type for the summarizeKnowledgeBase function.
 * - KnowledgeBaseSummarizerOutput - The return type for the summarizeKnowledgeBase function.
 */

import {ai} from '@/ai/genkit';
import {z} from 'genkit';

const KnowledgeBaseSummarizerInputSchema = z.object({
  content: z.string().describe('The content of the knowledge base (PDF text or article text).'),
});
export type KnowledgeBaseSummarizerInput = z.infer<typeof KnowledgeBaseSummarizerInputSchema>;

const KnowledgeBaseSummarizerOutputSchema = z.object({
  summary: z.string().describe('A concise summary of the knowledge base content.'),
});
export type KnowledgeBaseSummarizerOutput = z.infer<typeof KnowledgeBaseSummarizerOutputSchema>;

export async function summarizeKnowledgeBase(input: KnowledgeBaseSummarizerInput): Promise<KnowledgeBaseSummarizerOutput> {
  return summarizeKnowledgeBaseFlow(input);
}

const prompt = ai.definePrompt({
  name: 'knowledgeBaseSummarizerPrompt',
  input: {schema: KnowledgeBaseSummarizerInputSchema},
  output: {schema: KnowledgeBaseSummarizerOutputSchema},
  prompt: `You are an expert content summarizer.  Please provide a concise and accurate summary of the following knowledge base content:\n\nContent: {{{content}}}`,
});

const summarizeKnowledgeBaseFlow = ai.defineFlow(
  {
    name: 'summarizeKnowledgeBaseFlow',
    inputSchema: KnowledgeBaseSummarizerInputSchema,
    outputSchema: KnowledgeBaseSummarizerOutputSchema,
  },
  async input => {
    const {output} = await prompt(input);
    return output!;
  }
);
