
"use client";

import { useState } from "react";
import { type UseFormReturn } from "react-hook-form";
import { z } from "zod";
import { generateAgentPersona } from "@/ai/flows/agent-persona-generator";
import { generateKnowledgeBase } from "@/ai/flows/knowledge-base-generator";
import { useToast } from "@/hooks/use-toast";

import { Button } from "@/components/ui/button";
import {
  Form,
  FormControl,
  FormDescription,
  FormField,
  FormItem,
  FormLabel,
  FormMessage,
} from "@/components/ui/form";
import { Input } from "@/components/ui/input";
import { Textarea } from "@/components/ui/textarea";
import { Card, CardContent, CardHeader, CardTitle, CardDescription as CardDescriptionComponent } from "@/components/ui/card";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Sparkles, UploadCloud, Link as LinkIcon, FileText, Loader2, Settings, CreditCard, Percent, Image as ImageIcon, Video, UserPlus, FileQuestion, ShieldCheck } from "lucide-react";
import { Switch } from "@/components/ui/switch";
import { RadioGroup, RadioGroupItem } from "@/components/ui/radio-group";

export const formSchema = z.object({
  agentName: z.string().min(3, "Nome do agente deve ter no mínimo 3 caracteres."),
  agentDescription: z.string().min(10, "Descrição deve ter no mínimo 10 caracteres."),
  bannerImage: z.any().optional(),
  persona: z.string().min(10, "Persona deve ter no mínimo 10 caracteres."),
  toneOfVoice: z.string().min(5, "Tom de voz deve ter no mínimo 5 caracteres."),
  knowledgeBaseText: z.string().optional(),
  knowledgeBaseUrl: z.string().url().optional().or(z.literal('')),
  customInstructions: z.string().optional(),
  inputChannels: z.object({
    chatEmbedding: z.boolean().optional(),
    whatsapp: z.boolean().optional(),
    websiteUrl: z.boolean().optional(),
    googleDrive: z.boolean().optional(),
    googleSheets: z.boolean().optional(),
    json: z.boolean().optional(),
  }),
  outputChannels: z.object({
    chatEmbedding: z.boolean().optional(),
    whatsapp: z.boolean().optional(),
    websiteUrl: z.boolean().optional(),
    googleDrive: z.boolean().optional(),
    googleSheets: z.boolean().optional(),
    json: z.boolean().optional(),
  }),
  marketplaceDescription: z.string().optional(),
  marketplaceImages: z.any().optional(),
  marketplaceVideos: z.any().optional(),
  supportLink: z.string().url().optional().or(z.literal('')),
  termsOfUseLink: z.string().url().optional().or(z.literal('')),
  pricingModel: z.enum(["lifetime", "subscription"], {
    required_error: "Você precisa selecionar um modelo de precificação.",
  }),
  price: z.coerce.number().min(0, "O preço deve ser um número positivo.").optional(),
  affiliateEnabled: z.boolean().optional(),
  affiliateCommission: z.coerce.number().min(0).max(100).optional(),
});

interface CreateAgentFormProps {
    form: UseFormReturn<z.infer<typeof formSchema>>;
    step: number;
}

export function CreateAgentForm({ form, step }: CreateAgentFormProps) {
  const [isGeneratingPersona, setIsGeneratingPersona] = useState(false);
  const [isGeneratingKnowledge, setIsGeneratingKnowledge] = useState(false);
  const { toast } = useToast();
  const affiliateEnabled = form.watch("affiliateEnabled");

  const handleGeneratePersona = async () => {
    const agentDescription = form.getValues("agentDescription");
    if (!agentDescription) {
      form.setError("agentDescription", {
        type: "manual",
        message: "Por favor, forneça uma descrição para gerar uma persona.",
      });
      return;
    }

    setIsGeneratingPersona(true);
    try {
      const result = await generateAgentPersona({ agentDescription });
      form.setValue("persona", result.persona, { shouldValidate: true });
      form.setValue("toneOfVoice", result.toneOfVoice, { shouldValidate: true });
      toast({
        title: "Personalidade Gerada!",
        description: "A persona e o tom de voz do seu agente foram criados.",
      });
    } catch (error) {
      console.error("Falha ao gerar persona:", error);
      toast({
        variant: "destructive",
        title: "Falha na Geração",
        description: "Não foi possível gerar a personalidade. Tente novamente.",
      });
    } finally {
      setIsGeneratingPersona(false);
    }
  };

  const handleGenerateKnowledge = async () => {
    const agentDescription = form.getValues("agentDescription");
     if (!agentDescription) {
      form.setError("agentDescription", {
        type: "manual",
        message: "Por favor, forneça uma descrição para gerar a base de conhecimento.",
      });
      return;
    }
    setIsGeneratingKnowledge(true);
    try {
      const result = await generateKnowledgeBase({ agentDescription });
      form.setValue("knowledgeBaseText", result.knowledgeBaseText, { shouldValidate: true });
      toast({
        title: "Base de Conhecimento Gerada!",
        description: "Um texto inicial foi criado com base na descrição do seu agente.",
      });
    } catch (error) {
        console.error("Falha ao gerar conhecimento:", error);
        toast({
            variant: "destructive",
            title: "Falha na Geração",
            description: "Não foi possível gerar a base de conhecimento. Tente novamente.",
        });
    } finally {
        setIsGeneratingKnowledge(false);
    }
  }

  return (
      <>
        {step === 1 && (
             <div className="space-y-8">
                <Card>
                    <CardHeader>
                        <CardTitle className="font-headline text-2xl">Identidade do Agente</CardTitle>
                        <CardDescriptionComponent>Forneça as informações básicas do seu agente.</CardDescriptionComponent>
                    </CardHeader>
                    <CardContent className="space-y-6">
                        <FormField
                        control={form.control}
                        name="agentName"
                        render={({ field }) => (
                            <FormItem>
                            <FormLabel>Nome do Agente</FormLabel>
                            <FormControl>
                                <Input placeholder="ex: Mestre de Marketing" {...field} />
                            </FormControl>
                            <FormMessage />
                            </FormItem>
                        )}
                        />
                        <FormField
                        control={form.control}
                        name="agentDescription"
                        render={({ field }) => (
                            <FormItem>
                            <FormLabel>Descrição do Agente</FormLabel>
                            <FormControl>
                                <Textarea
                                placeholder="ex: Uma IA que gera textos criativos para marketing, analisa tendências e sugere campanhas..."
                                {...field}
                                rows={4}
                                />
                            </FormControl>
                             <FormDescription>
                                Descreva em detalhes o que seu agente faz. Isso é crucial para a geração da personalidade e conhecimento.
                              </FormDescription>
                            <FormMessage />
                            </FormItem>
                        )}
                        />
                         <FormField
                            control={form.control}
                            name="bannerImage"
                            render={({ field }) => (
                                <FormItem>
                                    <FormLabel>Banner do Agente</FormLabel>
                                    <FormDescription>Faça o upload de uma imagem 600x400px para o card do seu agente no marketplace.</FormDescription>
                                    <FormControl>
                                        <div className="flex flex-col items-center justify-center p-8 border-2 border-dashed rounded-lg text-center cursor-pointer hover:bg-accent/50 transition-colors">
                                            <UploadCloud className="h-12 w-12 text-muted-foreground"/>
                                            <p className="mt-4 text-sm font-semibold text-foreground">Clique para fazer upload ou arraste e solte</p>
                                            <p className="text-xs text-muted-foreground mt-1">PNG, JPG ou GIF (Recom. 600x400px)</p>
                                            <Input type="file" className="sr-only" {...field} />
                                        </div>
                                    </FormControl>
                                    <FormMessage />
                                </FormItem>
                            )}
                        />
                    </CardContent>
                </Card>

                <Card>
                    <CardHeader>
                        <div>
                            <CardTitle className="font-headline text-2xl">Personalidade</CardTitle>
                            <CardDescriptionComponent>Defina a persona e o tom de voz do seu agente, ou gere com IA.</CardDescriptionComponent>
                        </div>
                    </CardHeader>
                    <CardContent className="space-y-6">
                        <FormField
                        control={form.control}
                        name="persona"
                        render={({ field }) => (
                            <FormItem>
                               <div className="flex justify-between items-center">
                                 <FormLabel>Persona</FormLabel>
                                 <Button type="button" variant="ghost" size="sm" onClick={handleGeneratePersona} disabled={isGeneratingPersona}>
                                     {isGeneratingPersona ? <Loader2 className="mr-2 h-4 w-4 animate-spin" /> : <Sparkles className="mr-2 h-4 w-4" />}
                                     Gerar
                                 </Button>
                               </div>
                                <FormControl>
                                    <Textarea
                                    placeholder="Descreva a personalidade do seu agente. É formal, espirituoso, empático?"
                                    className="min-h-[120px]"
                                    {...field}
                                    />
                                </FormControl>
                                <FormMessage />
                            </FormItem>
                        )}
                        />
                        <FormField
                        control={form.control}
                        name="toneOfVoice"
                        render={({ field }) => (
                            <FormItem>
                                 <div className="flex justify-between items-center">
                                    <FormLabel>Tom de Voz</FormLabel>
                                    <Button type="button" variant="ghost" size="sm" onClick={handleGeneratePersona} disabled={isGeneratingPersona}>
                                        {isGeneratingPersona ? <Loader2 className="mr-2 h-4 w-4 animate-spin" /> : <Sparkles className="mr-2 h-4 w-4" />}
                                        Gerar
                                    </Button>
                                 </div>
                                <FormControl>
                                    <Textarea
                                    placeholder="Defina o estilo de fala do agente. Use frases curtas, emojis, tom formal, etc."
                                    {...field}
                                    />
                                </FormControl>
                                <FormMessage />
                            </FormItem>
                        )}
                        />
                    </CardContent>
                </Card>
            </div>
        )}
        {step === 2 && (
             <div className="space-y-8">
                <Card>
                    <CardHeader>
                        <div>
                            <CardTitle className="font-headline text-2xl">Base de Conhecimento</CardTitle>
                            <CardDescriptionComponent>
                            Forneça as informações que seu agente usará para responder. Você pode colar, gerar com IA, fazer upload ou importar de um link.
                            </CardDescriptionComponent>
                        </div>
                    </CardHeader>
                    <CardContent>
                        <Tabs defaultValue="text" className="w-full">
                        <TabsList className="grid w-full grid-cols-3">
                            <TabsTrigger value="text"><FileText className="mr-2 h-4 w-4"/>Texto</TabsTrigger>
                            <TabsTrigger value="upload"><UploadCloud className="mr-2 h-4 w-4"/>Upload de PDF</TabsTrigger>
                            <TabsTrigger value="link"><LinkIcon className="mr-2 h-4 w-4"/>Link</TabsTrigger>
                        </TabsList>
                        <TabsContent value="text" className="mt-4">
                            <FormField
                                control={form.control}
                                name="knowledgeBaseText"
                                render={({ field }) => (
                                    <FormItem>
                                        <div className="flex justify-between items-center mb-2">
                                            <FormLabel>Cole o Texto</FormLabel>
                                             <Button type="button" variant="ghost" size="sm" onClick={handleGenerateKnowledge} disabled={isGeneratingKnowledge}>
                                                {isGeneratingKnowledge ? <Loader2 className="mr-2 h-4 w-4 animate-spin" /> : <Sparkles className="mr-2 h-4 w-4" />}
                                                {isGeneratingKnowledge ? "Gerando..." : "Gerar com IA"}
                                            </Button>
                                        </div>
                                        <FormControl>
                                            <Textarea
                                            placeholder="Cole o conteúdo da sua base de conhecimento aqui ou gere com IA."
                                            className="min-h-[250px]"
                                            {...field}
                                            />
                                        </FormControl>
                                        <FormMessage />
                                    </FormItem>
                                )}
                                />
                        </TabsContent>
                        <TabsContent value="upload" className="mt-4">
                            <div className="flex flex-col items-center justify-center p-10 border-2 border-dashed rounded-lg text-center">
                                <UploadCloud className="h-12 w-12 text-muted-foreground"/>
                                <p className="mt-4 text-sm font-semibold text-foreground">Arraste e solte um arquivo PDF</p>
                                <p className="text-xs text-muted-foreground mt-1">ou clique para selecionar</p>
                                <Button type="button" variant="outline" className="mt-4">Selecionar Arquivo</Button>
                            </div>
                        </TabsContent>
                        <TabsContent value="link" className="mt-4">
                            <FormField
                                control={form.control}
                                name="knowledgeBaseUrl"
                                render={({ field }) => (
                                    <FormItem>
                                    <FormLabel>Importar de URL</FormLabel>
                                    <FormControl>
                                        <Input placeholder="https://exemplo.com/conhecimento" {...field} />
                                    </FormControl>
                                    <FormDescription>
                                        Nós vamos extrair o conteúdo desta página.
                                    </FormDescription>
                                    <FormMessage />
                                    </FormItem>
                                )}
                                />
                        </TabsContent>
                        </Tabs>
                    </CardContent>
                </Card>

                <Card>
                <CardHeader>
                    <CardTitle className="font-headline text-2xl">Instruções Customizadas</CardTitle>
                    <CardDescriptionComponent>Forneça regras e diretrizes específicas para o comportamento do seu agente.</CardDescriptionComponent>
                </CardHeader>
                <CardContent>
                    <FormField
                    control={form.control}
                    name="customInstructions"
                    render={({ field }) => (
                        <FormItem>
                        <FormControl>
                            <Textarea
                            placeholder="Ex: Sempre responda em Português. Nunca mencione concorrentes. Sempre inclua um call-to-action para o nosso site."
                            className="min-h-[150px]"
                            {...field}
                            />
                        </FormControl>
                        <FormMessage />
                        </FormItem>
                    )}
                    />
                </CardContent>
                </Card>
            </div>
        )}
        {step === 3 && (
            <div className="space-y-8">
                <Card>
                    <CardHeader>
                        <CardTitle className="font-headline text-2xl">Canais de Entrada</CardTitle>
                        <CardDescriptionComponent>Selecione como seu agente receberá informações.</CardDescriptionComponent>
                    </CardHeader>
                    <CardContent className="grid grid-cols-1 md:grid-cols-2 gap-6">
                        <FormField
                            control={form.control}
                            name="inputChannels.chatEmbedding"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                    <div className="space-y-0.5">
                                        <FormLabel className="text-base">Chat Embutido</FormLabel>
                                        <FormDescription>Permite que o agente interaja em um site.</FormDescription>
                                    </div>
                                    <div className="flex items-center gap-2">
                                        <FormControl>
                                            <Switch checked={field.value} onCheckedChange={field.onChange} />
                                        </FormControl>
                                        <Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button>
                                    </div>
                                </FormItem>
                            )}
                        />
                        <FormField
                            control={form.control}
                            name="inputChannels.whatsapp"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                    <div className="space-y-0.5">
                                        <FormLabel className="text-base">WhatsApp</FormLabel>
                                        <FormDescription>Conecte o agente a um número de WhatsApp.</FormDescription>
                                    </div>
                                    <div className="flex items-center gap-2">
                                        <FormControl>
                                            <Switch checked={field.value} onCheckedChange={field.onChange} />
                                        </FormControl>
                                        <Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button>
                                    </div>
                                </FormItem>
                            )}
                        />
                         <FormField
                            control={form.control}
                            name="inputChannels.websiteUrl"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                    <div className="space-y-0.5">
                                        <FormLabel className="text-base">URL de Site</FormLabel>
                                        <FormDescription>Use o conteúdo de um site como conhecimento.</FormDescription>
                                    </div>
                                    <div className="flex items-center gap-2">
                                        <FormControl>
                                            <Switch checked={field.value} onCheckedChange={field.onChange} />
                                        </FormControl>
                                        <Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button>
                                    </div>
                                </FormItem>
                            )}
                        />
                         <FormField
                            control={form.control}
                            name="inputChannels.googleDrive"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                    <div className="space-y-0.5">
                                        <FormLabel className="text-base">Google Drive</FormLabel>
                                        <FormDescription>Importe documentos do Google Drive.</FormDescription>
                                    </div>
                                    <div className="flex items-center gap-2">
                                        <FormControl>
                                            <Switch checked={field.value} onCheckedChange={field.onChange} />
                                        </FormControl>
                                        <Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button>
                                    </div>
                                </FormItem>
                            )}
                        />
                         <FormField
                            control={form.control}
                            name="inputChannels.googleSheets"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                    <div className="space-y-0.5">
                                        <FormLabel className="text-base">Google Sheets</FormLabel>
                                        <FormDescription>Importe dados de uma planilha do Google.</FormDescription>
                                    </div>
                                    <div className="flex items-center gap-2">
                                        <FormControl>
                                            <Switch checked={field.value} onCheckedChange={field.onChange} />
                                        </FormControl>
                                        <Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button>
                                    </div>
                                </FormItem>
                            )}
                        />
                         <FormField
                            control={form.control}
                            name="inputChannels.json"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                    <div className="space-y-0.5">
                                        <FormLabel className="text-base">Dados JSON</FormLabel>
                                        <FormDescription>Forneça dados através de uma estrutura JSON.</FormDescription>
                                    </div>
                                    <div className="flex items-center gap-2">
                                        <FormControl>
                                            <Switch checked={field.value} onCheckedChange={field.onChange} />
                                        </FormControl>
                                        <Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button>
                                    </div>
                                </FormItem>
                            )}
                        />
                    </CardContent>
                </Card>
                 <Card>
                    <CardHeader>
                        <CardTitle className="font-headline text-2xl">Canais de Saída</CardTitle>
                        <CardDescriptionComponent>Selecione como seu agente enviará informações.</CardDescriptionComponent>
                    </CardHeader>
                     <CardContent className="grid grid-cols-1 md:grid-cols-2 gap-6">
                       <FormField
                            control={form.control}
                            name="outputChannels.chatEmbedding"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                    <div className="space-y-0.5">
                                        <FormLabel className="text-base">Chat Embutido</FormLabel>
                                        <FormDescription>Envie respostas para um chat de site.</FormDescription>
                                    </div>
                                    <div className="flex items-center gap-2">
                                        <FormControl>
                                            <Switch checked={field.value} onCheckedChange={field.onChange} />
                                        </FormControl>
                                        <Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button>
                                    </div>
                                </FormItem>
                            )}
                        />
                         <FormField
                            control={form.control}
                            name="outputChannels.whatsapp"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                    <div className="space-y-0.5">
                                        <FormLabel className="text-base">WhatsApp</FormLabel>
                                        <FormDescription>Envie mensagens para um número de WhatsApp.</FormDescription>
                                    </div>
                                    <div className="flex items-center gap-2">
                                        <FormControl>
                                            <Switch checked={field.value} onCheckedChange={field.onChange} />
                                        </FormControl>
                                        <Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button>
                                    </div>
                                </FormItem>
                            )}
                        />
                         <FormField
                            control={form.control}
                            name="outputChannels.websiteUrl"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                    <div className="space-y-0.5">
                                        <FormLabel className="text-base">URL de Site (Webhook)</FormLabel>
                                        <FormDescription>Envie dados para uma URL externa.</FormDescription>
                                    </div>
                                    <div className="flex items-center gap-2">
                                        <FormControl>
                                            <Switch checked={field.value} onCheckedChange={field.onChange} />
                                        </FormControl>
                                        <Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button>
                                    </div>
                                </FormItem>
                            )}
                        />
                         <FormField
                            control={form.control}
                            name="outputChannels.googleDrive"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                    <div className="space-y-0.5">
                                        <FormLabel className="text-base">Google Drive</FormLabel>
                                        <FormDescription>Salve documentos e arquivos no Drive.</FormDescription>
                                    </div>
                                    <div className="flex items-center gap-2">
                                        <FormControl>
                                            <Switch checked={field.value} onCheckedChange={field.onChange} />
                                        </FormControl>
                                        <Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button>
                                    </div>
                                </FormItem>
                            )}
                        />
                         <FormField
                            control={form.control}
                            name="outputChannels.googleSheets"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                    <div className="space-y-0.5">
                                        <FormLabel className="text-base">Google Sheets</FormLabel>
                                        <FormDescription>Exporte dados para uma planilha do Google.</FormDescription>
                                    </div>
                                    <div className="flex items-center gap-2">
                                        <FormControl>
                                            <Switch checked={field.value} onCheckedChange={field.onChange} />
                                        </FormControl>
                                        <Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button>
                                    </div>
                                </FormItem>
                            )}
                        />
                         <FormField
                            control={form.control}
                            name="outputChannels.json"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                    <div className="space-y-0.5">
                                        <FormLabel className="text-base">Dados JSON</FormLabel>
                                        <FormDescription>Retorne dados em uma estrutura JSON.</FormDescription>
                                    </div>
                                    <div className="flex items-center gap-2">
                                        <FormControl>
                                            <Switch checked={field.value} onCheckedChange={field.onChange} />
                                        </FormControl>
                                        <Button size="icon" variant="ghost" disabled={!field.value}><Settings className="h-4 w-4"/></Button>
                                    </div>
                                </FormItem>
                            )}
                        />
                    </CardContent>
                </Card>
            </div>
        )}
        {step === 4 && (
            <div className="space-y-8">
                <Card>
                    <CardHeader>
                        <CardTitle className="font-headline text-2xl">Página do Marketplace</CardTitle>
                        <CardDescriptionComponent>Configure como seu agente será exibido para os clientes.</CardDescriptionComponent>
                    </CardHeader>
                    <CardContent className="space-y-6">
                        <FormField
                        control={form.control}
                        name="marketplaceDescription"
                        render={({ field }) => (
                            <FormItem>
                                <FormLabel>Descrição Detalhada</FormLabel>
                                <FormControl>
                                    <div className="w-full rounded-md border border-input bg-background p-3">
                                        {/* This would be a rich text editor */}
                                        <Textarea
                                            placeholder="Descreva seu agente em detalhes. Use formatação para destacar recursos e benefícios..."
                                            className="min-h-[200px] border-0 p-0 focus-visible:ring-0 focus-visible:ring-offset-0"
                                            {...field}
                                        />
                                    </div>
                                </FormControl>
                                <FormDescription>
                                Este conteúdo será usado na página de vendas do seu agente.
                                </FormDescription>
                                <FormMessage />
                            </FormItem>
                        )}
                        />
                        <div className="grid grid-cols-1 md:grid-cols-2 gap-6">
                             <FormField
                                control={form.control}
                                name="marketplaceImages"
                                render={({ field }) => (
                                    <FormItem>
                                        <FormLabel>Imagens Promocionais</FormLabel>
                                        <FormControl>
                                            <div className="flex flex-col items-center justify-center p-8 border-2 border-dashed rounded-lg text-center cursor-pointer hover:bg-accent/50 transition-colors">
                                                <ImageIcon className="h-12 w-12 text-muted-foreground"/>
                                                <p className="mt-4 text-sm font-semibold text-foreground">Clique para fazer upload de imagens</p>
                                                <p className="text-xs text-muted-foreground mt-1">PNG, JPG, GIF</p>
                                                <Input type="file" multiple className="sr-only" {...field} />
                                            </div>
                                        </FormControl>
                                        <FormMessage />
                                    </FormItem>
                                )}
                            />
                             <FormField
                                control={form.control}
                                name="marketplaceVideos"
                                render={({ field }) => (
                                    <FormItem>
                                        <FormLabel>Vídeo Demonstrativo</FormLabel>
                                        <FormControl>
                                            <div className="flex flex-col items-center justify-center p-8 border-2 border-dashed rounded-lg text-center cursor-pointer hover:bg-accent/50 transition-colors">
                                                <Video className="h-12 w-12 text-muted-foreground"/>
                                                <p className="mt-4 text-sm font-semibold text-foreground">Clique para fazer upload de vídeos</p>
                                                <p className="text-xs text-muted-foreground mt-1">MP4, WEBM</p>
                                                <Input type="file" multiple className="sr-only" {...field} />
                                            </div>
                                        </FormControl>
                                        <FormMessage />
                                    </FormItem>
                                )}
                            />
                        </div>
                         <FormField
                            control={form.control}
                            name="supportLink"
                            render={({ field }) => (
                                <FormItem>
                                    <FormLabel>Link de Suporte</FormLabel>
                                     <div className="flex items-center gap-2">
                                        <FileQuestion className="h-5 w-5 text-muted-foreground" />
                                        <FormControl>
                                            <Input placeholder="https://exemplo.com/suporte" {...field} />
                                        </FormControl>
                                    </div>
                                    <FormMessage />
                                </FormItem>
                            )}
                        />
                        <FormField
                            control={form.control}
                            name="termsOfUseLink"
                            render={({ field }) => (
                                <FormItem>
                                    <FormLabel>Link de Termos de Uso</FormLabel>
                                    <div className="flex items-center gap-2">
                                        <ShieldCheck className="h-5 w-5 text-muted-foreground" />
                                        <FormControl>
                                            <Input placeholder="https://exemplo.com/termos" {...field} />
                                        </FormControl>
                                    </div>
                                    <FormMessage />
                                </FormItem>
                            )}
                        />
                    </CardContent>
                </Card>
                 <Card>
                    <CardHeader>
                        <CardTitle className="font-headline text-2xl flex items-center gap-2"><UserPlus /> Programa de Afiliados</CardTitle>
                         <CardDescriptionComponent>
                           Permita que outros vendam seu agente em troca de uma comissão.
                        </CardDescriptionComponent>
                    </CardHeader>
                    <CardContent className="space-y-6">
                        <FormField
                            control={form.control}
                            name="affiliateEnabled"
                            render={({ field }) => (
                                <FormItem className="flex flex-row items-center justify-between rounded-lg border p-4">
                                    <div className="space-y-0.5">
                                        <FormLabel className="text-base">Ativar Afiliação</FormLabel>
                                        <FormDescription>Permitir que outros promovam seu agente.</FormDescription>
                                    </div>
                                    <FormControl>
                                        <Switch checked={field.value} onCheckedChange={field.onChange} />
                                    </FormControl>
                                </FormItem>
                            )}
                        />
                        {affiliateEnabled && (
                             <FormField
                                control={form.control}
                                name="affiliateCommission"
                                render={({ field }) => (
                                    <FormItem>
                                    <FormLabel>Comissão do Afiliado</FormLabel>
                                    <div className="relative">
                                        <FormControl>
                                            <Input type="number" placeholder="20" className="pr-8" {...field} />
                                        </FormControl>
                                        <span className="absolute inset-y-0 right-0 flex items-center pr-3 text-muted-foreground">%</span>
                                    </div>
                                    <FormDescription>A porcentagem que os afiliados receberão por cada venda.</FormDescription>
                                    <FormMessage />
                                    </FormItem>
                                )}
                            />
                        )}
                    </CardContent>
                </Card>
            </div>
        )}
        {step === 5 && (
             <div className="space-y-8">
                <Card>
                    <CardHeader>
                        <CardTitle className="font-headline text-2xl">Precificação</CardTitle>
                        <CardDescriptionComponent>Defina um preço para o seu agente.</CardDescriptionComponent>
                    </CardHeader>
                    <CardContent className="space-y-6">
                        <FormField
                        control={form.control}
                        name="pricingModel"
                        render={({ field }) => (
                            <FormItem className="space-y-3">
                            <FormLabel>Modelo de Precificação</FormLabel>
                            <FormControl>
                                <RadioGroup
                                onValueChange={field.onChange}
                                defaultValue={field.value}
                                className="flex flex-col space-y-1"
                                >
                                <FormItem className="flex items-center space-x-3 space-y-0">
                                    <FormControl>
                                    <RadioGroupItem value="lifetime" />
                                    </FormControl>
                                    <FormLabel className="font-normal">
                                    Acesso Vitalício (pagamento único)
                                    </FormLabel>
                                </FormItem>
                                <FormItem className="flex items-center space-x-3 space-y-0">
                                    <FormControl>
                                    <RadioGroupItem value="subscription" />
                                    </FormControl>
                                    <FormLabel className="font-normal">
                                     Assinatura (recorrente mensal)
                                    </FormLabel>
                                </FormItem>
                                </RadioGroup>
                            </FormControl>
                            <FormMessage />
                            </FormItem>
                        )}
                        />
                         <FormField
                            control={form.control}
                            name="price"
                            render={({ field }) => (
                                <FormItem>
                                <FormLabel>Preço</FormLabel>
                                <div className="relative">
                                    <span className="absolute inset-y-0 left-0 flex items-center pl-3 text-muted-foreground">R$</span>
                                    <FormControl>
                                        <Input type="number" placeholder="49,99" className="pl-8" {...field} />
                                    </FormControl>
                                     <span className="absolute inset-y-0 right-0 flex items-center pr-3 text-muted-foreground">
                                        {form.getValues("pricingModel") === "subscription" ? "/ mês" : ""}
                                    </span>
                                </div>
                                <FormMessage />
                                </FormItem>
                            )}
                        />
                    </CardContent>
                </Card>
                 <Card>
                    <CardHeader>
                        <CardTitle className="font-headline text-2xl flex items-center gap-2"><CreditCard /> Pagamentos</CardTitle>
                        <CardDescriptionComponent>
                            Os pagamentos são processados via Stripe.
                        </CardDescriptionComponent>
                    </CardHeader>
                    <CardContent className="space-y-4">
                        <div className="flex items-center justify-between rounded-lg border p-4">
                            <div>
                                <p className="font-semibold">Stripe Connect</p>
                                <p className="text-sm text-muted-foreground">Conta conectada: acct_...****</p>
                            </div>
                            <Button variant="outline">Gerenciar</Button>
                        </div>
                        <div className="flex items-center justify-between rounded-lg border p-4 bg-muted/20">
                           <div className="flex items-center gap-2 text-sm text-muted-foreground">
                             <Percent className="h-4 w-4" />
                             <p>Taxa de transação da plataforma</p>
                           </div>
                           <p className="text-sm font-semibold">10%</p>
                        </div>
                    </CardContent>
                </Card>
            </div>
        )}
      </>
  );
}

    