
"use client";

import Image from "next/image";
import type { Agent } from "@/lib/types";
import {
  Card,
  CardContent,
  CardDescription,
  CardFooter,
  CardHeader,
  CardTitle,
} from "@/components/ui/card";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Star, Eye, EyeOff, KeyRound, BarChart, Pencil, ShoppingBag, Edit, Settings, Share2 } from "lucide-react";
import { useState } from "react";
import { cn } from "@/lib/utils";
import Link from "next/link";
import { usePathname } from "next/navigation";


interface AgentCardProps {
  agent: Agent;
  variant?: "created" | "purchased" | "marketplace" | "draft";
}

export function AgentCard({ agent, variant = "marketplace" }: AgentCardProps) {
    const [isKeyVisible, setIsKeyVisible] = useState(false);
    const pathname = usePathname();
    const locale = pathname.split('/')[1] || 'pt';

  return (
    <Card className={cn(
        "flex flex-col h-full overflow-hidden transition-all duration-300 hover:shadow-xl hover:-translate-y-1 bg-card/80",
        variant === "created" && "border-primary/20",
        variant === "draft" && "opacity-70 border-dashed"
    )}>
      <CardHeader className="p-0">
        <Link href={`/${locale}/marketplace/${agent.id}`}>
            <div className="relative">
            <Image
                src={agent.imageUrl}
                alt={agent.name}
                width={600}
                height={400}
                className="object-cover w-full h-48"
                data-ai-hint="abstract technology"
            />
            <div className="absolute top-2 right-2 flex gap-2">
                    {variant === 'created' && <Badge variant="secondary">Criado por você</Badge>}
                    {variant === 'purchased' && <Badge variant="default">Comprado</Badge>}
                    {variant === 'marketplace' && <Badge variant="secondary">{agent.category}</Badge>}
                    {variant === 'draft' && <Badge variant="outline">Rascunho</Badge>}
            </div>
            </div>
        </Link>
      </CardHeader>
      <CardContent className="p-4 flex-grow">
        <CardTitle className="text-lg font-bold font-headline mb-2">{agent.name}</CardTitle>
        <CardDescription className="text-sm text-muted-foreground line-clamp-3">
          {agent.description}
        </CardDescription>
      </CardContent>
      <CardFooter className="p-4 pt-0 flex flex-col items-start gap-4">
        {variant === 'marketplace' && (
             <>
                <div className="flex items-center justify-between w-full">
                    <div className="flex items-center gap-2">
                        <Avatar className="h-8 w-8">
                            <AvatarImage src={agent.creatorImage} alt={agent.creatorName} />
                            <AvatarFallback>{agent.creatorName.charAt(0)}</AvatarFallback>
                        </Avatar>
                        <span className="text-sm font-medium">{agent.creatorName}</span>
                    </div>
                    <div className="flex items-center gap-1">
                        <Star className="w-4 h-4 text-amber-400 fill-amber-400" />
                        <span className="text-sm font-bold">{agent.rating}</span>
                        <span className="text-xs text-muted-foreground">({agent.reviews})</span>
                    </div>
                </div>
                <div className="flex items-center justify-between w-full">
                    <p className="text-2xl font-bold font-headline text-primary">
                        ${agent.price.toFixed(2)}
                    </p>
                    <Button asChild>
                       <Link href={`/${locale}/marketplace/${agent.id}`}>Ver Detalhes</Link>
                    </Button>
                </div>
            </>
        )}
        {variant === 'purchased' && (
             <div className="w-full pt-4 border-t space-y-2">
                <div className="flex justify-between items-center">
                    <div className="text-xs text-muted-foreground">Sua Chave</div>
                    <Button variant="outline" size="sm"><Settings className="mr-2 h-4 w-4"/>Configurar</Button>
                </div>
                <div className="flex items-center gap-2">
                    <div className={cn(
                        "flex-grow rounded-md bg-muted px-3 py-2 text-sm font-mono",
                        !isKeyVisible && "blur-sm"
                    )}>
                        {isKeyVisible ? `sk-${agent.id}` : 'sk-****************'}
                    </div>
                    <Button variant="ghost" size="icon" onClick={() => setIsKeyVisible(!isKeyVisible)}>
                        {isKeyVisible ? <EyeOff /> : <Eye />}
                    </Button>
                </div>
            </div>
        )}
        {variant === 'created' && (
            <div className="flex w-full items-center justify-between">
                <div className="flex items-center gap-1 text-sm text-muted-foreground">
                    <ShoppingBag className="h-4 w-4" />
                    <span>{agent.reviews} Vendas</span>
                </div>
                <div className="flex gap-2">
                    <Button variant="outline" size="icon"><Share2 /></Button>
                    <Button variant="outline" size="icon"><Pencil /></Button>
                    <Button variant="outline" size="icon"><BarChart /></Button>
                </div>
            </div>
        )}
        {variant === 'draft' && (
            <div className="flex w-full items-center justify-end">
                 <Button variant="secondary" className="w-full">
                    <Edit className="mr-2 h-4 w-4"/>
                    Continuar Editando
                </Button>
            </div>
        )}
      </CardFooter>
    </Card>
  );
}
